/**
 * \file: AautoLayerManager.cpp
 *
 * \version: $Id:$
 *
 * \release: $Name:$
 *
 * <brief description>.
 * <detailed description>
 * \component: Android auto - test-control
 *
 * \author: Veeraiyan Chidambaram /RBEI/ECF3/ veeraiyan.chidambaram@in.bosch.com
 *
 *
 * \copyright (c) 2013 Advanced Driver Information Technology.
 * This code is developed by Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 * \see <related items>
 *
 * \history
 *
 ***********************************************************************/

#include <stdbool.h>
#include <unistd.h>

#include "AutoSmoketest.h"
#include "utils/AAutoLayerManager.h"
#include "Testbed.h"

LOG_IMPORT_CONTEXT(demo)

namespace adit { namespace aauto {

#define DEFAULT_LAYER_ID            3000
#define DEFAULT_SURFACE_ID          40
#define DEFAULT_DISPLAY_ID          1

// Screen/touch resolution
#define DEFAULT_RES_WIDTH       800
#define DEFAULT_RES_HEIGHT      480
#define DEFAULT_DEVICE_MAX      10

aautoLayerManager::aautoLayerManager(IDynamicConfiguration& inConfig)
{
    mConfig   = &inConfig;
    mIlmInit = false;
    mActiveDevice = "";

    if( AutoSmoketest::instance().getTestMode() == AUTOMATIC)
        //use display id from smoketest arguments
        mDisplayId  = AutoSmoketest::instance().getDisplayId();
    else
        //use display id from cfg file
        mDisplayId  = (t_ilm_display)mConfig->GetNumber("aauto-demo-display-id", DEFAULT_DISPLAY_ID);

    mLayerId    = (t_ilm_layer)mConfig->GetNumber("aauto-demo-layer-id", DEFAULT_LAYER_ID);
    mSurfaceId  = (t_ilm_surface)mConfig->GetNumber("aauto-demo-surface-id", DEFAULT_SURFACE_ID);
    mMaxDevices = (int32_t)mConfig->GetNumber("aauto-demo-devices-max", DEFAULT_DEVICE_MAX);
    mSharedSurface = (bool)mConfig->GetNumber("aauto-demo-share-surface", 1);
    mResolutionWidth = (t_ilm_uint)mConfig->GetNumber("aauto-demo-resolution-width",DEFAULT_RES_WIDTH);
    mResolutionHeight = (t_ilm_uint)mConfig->GetNumber("aauto-demo-resolution-height", DEFAULT_RES_HEIGHT);

    pthread_mutex_init(&orderMutex, NULL);
}

aautoLayerManager::~aautoLayerManager()
{
    pthread_mutex_destroy(&orderMutex);
}

bool aautoLayerManager::addMobileDevice(const std::string inDeviceSerial)
{
    bool ret = false;

    if(!mIlmInit)
    {
        if(init() == false)
        {
            return false;
        }
    }

    int free_device_index = getFreeDeviceIndex();
    if (free_device_index == -1)
    {
        LOG_ERROR((demo, "Could not get any free device index, session count exceeded"));

    } else {

        mobileDeviceSettings new_device_settings;
        new_device_settings.deviceIndex = free_device_index;
        new_device_settings.surfaceId = mSurfaceId + free_device_index;
        new_device_settings.size.srcWidth = DEFAULT_RES_WIDTH;
        new_device_settings.size.srcHeight = DEFAULT_RES_HEIGHT;
        new_device_settings.size.destWidth  = mResolutionWidth;
        new_device_settings.size.destHeight = mResolutionHeight;
        mMobileDevices.insert(std::pair<std::string, mobileDeviceSettings>(inDeviceSerial,
                new_device_settings));

        pthread_mutex_lock(&orderMutex);
        ret = refreshZOrder(inDeviceSerial);
        pthread_mutex_unlock(&orderMutex);
    }

    return ret;
}

void aautoLayerManager::deleteMobileDevice(const std::string inDeviceSerial)
{
    auto itr = mMobileDevices.find(inDeviceSerial);
    if(itr != mMobileDevices.end())
    {
        mMobileDevices.erase(itr);
        mActiveDevice = "";
        LOG_INFO((demo, "Unplugged device %s", inDeviceSerial.c_str()));

        if(mMobileDevices.size() != 0)
        {
            // reset Layer Order
            itr = mMobileDevices.begin();
            LOG_INFO((demo, "Bring device %s to front", itr->first.c_str()));

            pthread_mutex_lock(&orderMutex);
            refreshZOrder(itr->first);
            pthread_mutex_unlock(&orderMutex);
        }
    } else {
        LOG_ERROR((demo, "Could not delete mobile device with serial number %s.",
                   inDeviceSerial.c_str()));
    }
}

void aautoLayerManager::finalize()
{
    if (mIlmInit)
    {
        for(auto iter = mMobileDevices.begin(); iter != mMobileDevices.end(); )
        {
            iter = mMobileDevices.erase(iter);
        }

        if(Testbed::instance().getSurfaceID())
        {
            // don't do anything
        }
        else
        {
            removeLayer();
        }

        ilm_unregisterNotification();
        ilm_destroy();
        mIlmInit = false;
    }
}

bool aautoLayerManager::bringDeviceToFront(const int inDeviceIndex)
{
    bool ret = false;
    LOG_INFO((demo, "%s() DeviceIndex = %d", __FUNCTION__, inDeviceIndex));

    for (auto itr = mMobileDevices.begin(); itr != mMobileDevices.end(); itr++)
    {
        mobileDeviceSettings md_settings = itr->second;
        if(md_settings.deviceIndex == inDeviceIndex)
        {
            pthread_mutex_lock(&orderMutex);
            LOG_INFO((demo, "%s() Found settings for DeviceIndex = %d", __FUNCTION__, inDeviceIndex));
            ret = refreshZOrder(itr->first);
            pthread_mutex_unlock(&orderMutex);
            break;
        }
    }

    return ret;
}

t_ilm_layer aautoLayerManager::getLayerId() const
{
    return mLayerId;
}

t_ilm_surface aautoLayerManager::getSurfaceId(std::string inDeviceSerial) const
{
    t_ilm_surface ret_surface_id = -1;
    auto itr = mMobileDevices.find(inDeviceSerial);
    if( itr != mMobileDevices.end() ) {
        mobileDeviceSettings md_settings = itr->second;
        ret_surface_id = md_settings.surfaceId;
    }else{
        LOG_ERROR((demo, "Could not find device %s to return its surface ID",
                inDeviceSerial.c_str()));
    }

    return ret_surface_id;
}

t_ilm_surface aautoLayerManager::getTouchSurfaceId(std::string inDeviceSerial) const
{
    if (mSharedSurface)
    {
        return getSurfaceId(inDeviceSerial);
    }
    else
    {
        return getSurfaceId(inDeviceSerial) + 1;
    }
}

bool aautoLayerManager::init()
{
    if (ILM_SUCCESS != ilm_init())
    {
        LOG_ERROR((demo, "ilm_init failed"));
        return false;
    }

    mIlmInit = true;

    return createLayer(mResolutionWidth, mResolutionHeight);
}

bool aautoLayerManager::createLayer(const t_ilm_uint inSrcWidth, const t_ilm_uint inSrcHeight)
{

    if (ILM_SUCCESS != ilm_registerNotification(ilmNotifyCB, this))
    {
        LOG_ERROR((demo, "%s()  ilm_registerNotification failed", __func__));
        return false;
    }

    if(Testbed::instance().getTestbedMode())
    {
        // don't do anything
    }
    else
    {
        if (ILM_SUCCESS != ilm_layerCreateWithDimension(&mLayerId, inSrcWidth, inSrcHeight))
        {
            LOG_ERROR((demo, "%s()  ilm_layerCreateWithDimension for layer %d", __func__, mLayerId));
            return false;
        }

        /* workaroud for getting current active disaplay id */
        t_ilm_uint    screen_width   = -1;
        t_ilm_uint    screen_height  = -1;
        t_ilm_uint    display_width  = -1;
        t_ilm_uint    display_height = -1;
        t_ilm_display display_id     = -1;

        bool valid = false;
        int32_t ret = ILM_SUCCESS;

        /* try to get resolution of display represented by mDisplayId */
        ret = ilm_getScreenResolution(mDisplayId, &screen_width, &screen_height);
        if (ILM_SUCCESS != ret)
        {
            LOG_INFO((demo, "%s()  Could not get screen resolution of displayId %d (%d x %d). ret=%d",
                    __func__, mDisplayId, screen_width, screen_height, ret));
        }
        else
        {
            /* check if received width and height are valid */
            if ( ((screen_width != 0) && (screen_height != 0))
                 && ((screen_width != (unsigned int)-1) && (screen_height != (unsigned int)-1)) )
            {
                valid = true;
            }
            else
            {
                LOG_INFO((demo, "%s()  Got invalid screen resolution for displayId %d with width=%d, height=%d",
                        __func__, mDisplayId, screen_width, screen_height));
            }
        }

        if (true == getHighestResolution(&display_id, &display_width, &display_height))
        {
            if (true == valid)
            {
                if ( (display_width > screen_width) && (mDisplayId != display_id) )
                {
                    LOG_INFO((demo, "%s()  Found displayId %d with higher resolution (%d x %d). Take it.",
                            __func__, display_id, display_width, display_height));

                    mDisplayId = display_id;
                    screen_width = display_width;
                    screen_height = display_height;

                    valid = true;
                }
                else if ( (display_width == screen_width) && (mDisplayId != display_id) )
                {
                    LOG_INFO((demo, "%s()  Found different displayId %d with same resolution (%d x %d). Use configured displayId %d.",
                            __func__, display_id, display_width, display_height, mDisplayId));
                }
                else
                {
                    LOG_INFO((demo, "%s()  Configured displayId %d has highest resolution (%d x %d). Keep it",
                            __func__, mDisplayId, screen_width, screen_height));
                }
            }
            else
            {
                mDisplayId = display_id;
                screen_width = display_width;
                screen_height = display_height;

                valid = true;
                LOG_INFO((demo, "%s()  Take highest resolution of displayId %d with width=%d, height=%d",
                        __func__, mDisplayId, screen_width, screen_height));
            }
        }
        else
        {
            LOG_INFO((demo, "%s()  Could not get higher resolution. Keep configured displayId=%d with width=%d, height=%d",
                    __func__, mDisplayId, screen_width, screen_height));
        }

        /* valid resolution, keep it */
        if (true != valid)
        {
            LOG_ERROR((demo, "%s()  No valid display", __func__));
            return false; /* ==== leave function ==== */
        }

        t_ilm_uint origin_x = 0, origin_y = 0;
        if(screen_width > inSrcWidth)
        {
            origin_x = (screen_width - inSrcWidth) / 2;
        }
        if(screen_height > inSrcHeight)
        {
            origin_y = (screen_height - inSrcHeight) / 2;
        }

        if (ILM_SUCCESS != ilm_layerSetDestinationRectangle(mLayerId, origin_x, origin_y,
                                                            inSrcWidth, inSrcHeight))
        {
            LOG_ERROR((demo, "%s()  layerSetDestinationRectangle for layer %d", __func__, mLayerId));
            return false;
        }

        if (ILM_SUCCESS != ilm_layerSetVisibility(mLayerId, ILM_TRUE))
        {
            LOG_ERROR((demo, "%s()  ilm_layerSetVisibility failed for layer %d", __func__, mLayerId));
            return false;
        }

        if (ILM_SUCCESS != ilm_displaySetRenderOrder(mDisplayId, &mLayerId, 1))
        {
            LOG_ERROR((demo, "%s()  ilm_displaySetRenderOrder failed for display %d, layer %d",
                    __func__, mDisplayId, mLayerId));
            return false;
        }
    }

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((demo, "%s()  ilm_commitChanges failed", __func__));
        return false;
    }

    return true;
}

bool aautoLayerManager::getHighestResolution(t_ilm_display* outDisplayId, t_ilm_uint* outSrcWidth, t_ilm_uint* outSrcHeight)
{
    t_ilm_uint     display_width  = -1;
    t_ilm_uint     display_height = -1;
    t_ilm_display* display_ids    = NULL;
    t_ilm_uint     display_count  = 0;

    bool     valid = false;
    int32_t  ret   = ILM_SUCCESS;
    uint32_t i     = 0;

    ilm_getScreenIDs(&display_count, &display_ids);
    LOG_INFO((demo, "%s()  Available number of displays: %d", __func__, display_count));

    /* get displayId with highest resolution */
    while(i < display_count)
    {
        /* try to get resolution of display represented by display_ids[i]*/
        ret = ilm_getScreenResolution(display_ids[i], &display_width, &display_height);
        if (ILM_SUCCESS != ret)
        {
            LOG_INFO((demo, "%s()  Could not get screen resolution of displayId %d (width:%d, height:%d). ret=%d",
                    __func__, display_ids[i], display_width, display_height, ret));
        }
        else
        {
            /* check if received width and height are valid */
            if ( ((display_width != 0) && (display_height != 0))
                 && ((display_width != (unsigned int)-1) && (display_height != (unsigned int)-1)) )
            {
                if (*outDisplayId == (unsigned int)-1)
                {
                    /* first time that we get resolution. keep it */
                    *outSrcWidth = display_width;
                    *outSrcHeight = display_height;
                    *outDisplayId = display_ids[i];

                    valid = true;
                    LOG_INFO((demo, "%s()  Found first displayId %d with width:%u, height:%u. Keep it and continue",
                            __func__, *outDisplayId, *outSrcWidth, *outSrcHeight));
                }
                else
                {
                    /* check if new display resolution is higher than previous one */
                    if (display_width > *outSrcWidth)
                    {
                        /* valid resolution, keep it */
                        *outSrcWidth = display_width;
                        *outSrcHeight = display_height;
                        *outDisplayId = display_ids[i];

                        valid = true;
                        LOG_INFO((demo, "%s()  Found new displayId %d with width:%u, height:%u",
                                __func__, *outDisplayId, *outSrcWidth, *outSrcHeight));
                    }
                    else
                    {
                        LOG_INFO((demo, "%s()  displayId %d with width:%u, height:%u has no higher resolution than displayId %d (%u x %u)",
                                __func__, display_ids[i], display_width, display_height, *outDisplayId, *outSrcWidth, *outSrcHeight));
                    }
                }
            }
            else
            {
                LOG_INFO((demo, "%s()  Got invalid screen resolution for displayId=%d with width=%d, height=%d",
                        __func__, display_ids[i], display_width, display_height));
            }
        }
        i++;
    }
    if (NULL != display_ids)
    {
        free(display_ids);
        display_ids = NULL;
    }

    return valid;
}

int aautoLayerManager::getFreeDeviceIndex()
{
    int free_device_index = -1;
    bool taken_indexes[mMaxDevices];
    memset(taken_indexes, false, mMaxDevices*sizeof(bool));

    std::map<std::string, mobileDeviceSettings>::iterator itr;
    for (itr = mMobileDevices.begin(); itr != mMobileDevices.end(); itr++)
    {
        mobileDeviceSettings taken_md_settings = itr->second;
        taken_indexes[taken_md_settings.deviceIndex] = true;
        LOGD_DEBUG((demo, "Device index %d already taken", taken_md_settings.deviceIndex));
    }

    for(int ix = 0; ix < mMaxDevices; ix++)
    {
        if (taken_indexes[ix] == false)
        {
            free_device_index = ix;
            break;
        }
    }
    LOGD_DEBUG((demo, "Returning free device index %d", free_device_index));

    return free_device_index;
}

void aautoLayerManager::removeLayer()
{
    if (ILM_SUCCESS != ilm_layerSetVisibility(mLayerId, ILM_FALSE))
    {
        LOG_ERROR((demo, "ilm_layerSetVisibility failed for layer %d", mLayerId));
    }

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((demo, "ilm_commitChanges failed"));
    }

    LOGD_DEBUG((demo, "remove layer %d", mLayerId));
    ilm_layerRemove(mLayerId);

    if (ILM_SUCCESS != ilm_commitChanges())
    {
        LOG_ERROR((demo, "ilm_commitChanges failed"));
    }
}

bool aautoLayerManager::refreshZOrder(const std::string inDeviceSerial)
{
    if(Testbed::instance().getTestbedMode())
    {
        // don't do anything
    }
    else
    {
        auto itr = mMobileDevices.find(inDeviceSerial);
        if(itr != mMobileDevices.end())
        {
            t_ilm_surface new_active_surface = itr->second.surfaceId;
            if (mActiveDevice != inDeviceSerial)
            {
                if (ILM_SUCCESS != ilm_layerSetRenderOrder(mLayerId, &new_active_surface, 1))
                {
                    LOG_ERROR((demo, "Failed to bring surface %d to front.", new_active_surface));
                    return false;
                }

                if (ILM_SUCCESS != ilm_commitChanges())
                {
                    LOG_ERROR((demo, "ilm_commitChanges failed"));
                    return false;
                }
                mActiveDevice = inDeviceSerial;
            }

        } else {
            LOG_ERROR((demo, "Could not find the given mobile device: %s", inDeviceSerial.c_str()));
            return false;
        }
    }
    
    return true;;
}
void aautoLayerManager::ilmNotifyCB(ilmObjectType object, t_ilm_uint id, t_ilm_bool created, void *user_data)
{
    LOG_INFO((demo, "aautoLayerManager::%s() called ", __FUNCTION__));

    aautoLayerManager *alm = static_cast<aautoLayerManager*> (user_data);
    struct ilmSurfaceProperties sp;

    if (object == ILM_SURFACE)
    {
        if (created)
        {

            LOG_INFO((demo,"aautoLayerManager::%s() surface (id=%d) created", __FUNCTION__, id));
            ilm_getPropertiesOfSurface(id, &sp);

            LOG_INFO((demo,"aautoLayerManager::%s() origSourceWidth: %d  origSourceHeight: %d",
                    __FUNCTION__, sp.origSourceWidth, sp.origSourceHeight));
            if ((sp.origSourceWidth != 0) && (sp.origSourceHeight != 0))
            {
                if(Testbed::instance().getTestbedMode() && id != (unsigned int) Testbed::instance().getSurfaceID())
                {
                    // If testbed mode is turned on,
                    // then it don't do anything if surface id isn't video surface id.
                }
                else
                {
                    // surface is already configured
                    alm->configure_ilm_surface(id, sp.origSourceWidth, sp.origSourceHeight);
                }
            }
            else
            {
#ifndef IMX_DEMO
                if(Testbed::instance().getTestbedMode())
                {
                    // don't do anything
                }
                else
                {
                    // wait for configured event
                    ilm_surfaceAddNotification(id, &ilmSurfaceCB);
                    ilm_commitChanges();
                }
#else
                LOG_WARN((demo,"aautoLayerManager::%s() surface (id=%d) not configured. Configure notification not supported on IMX6",
                        __FUNCTION__, id));
#endif
            }

        }
        else if(!created)
        {
            LOG_INFO((demo,"aautoLayerManager::%s() surface (id=%d) destroyed", __FUNCTION__, id));

            if(Testbed::instance().getTestbedMode())
            {
                if(Testbed::instance().getDeviceState() == DISCONNECTED)
                {
                    Testbed::instance().adit_testbed_lost();
                    LOG_INFO((demo,"aautoLayerManager::%s - finished DISCONNECT", __FUNCTION__));
                }
                else
                {
                    Testbed::instance().adit_testbed_background();
                    LOG_INFO((demo,"aautoLayerManager::%s - finished CHANGE TO BACKGROUND", __FUNCTION__));
                }
            }
        }
    }
    else if (object == ILM_LAYER)
    {
        if (created)
            LOG_INFO((demo,"aautoLayerManager::%s() layer (id=%d) created", __FUNCTION__, id));
        else if(!created)
            LOG_INFO((demo,"aautoLayerManager::%s() layer (id=%d) destroyed", __FUNCTION__, id));
    }
}

void aautoLayerManager::configure_ilm_surface(t_ilm_uint id, t_ilm_uint width, t_ilm_uint height)
{
    LOG_INFO((demo, "aautoLayerManager::%s() called", __FUNCTION__));

    if (ILM_SUCCESS != ilm_surfaceSetDestinationRectangle(id, 0, 0, width, height))
        LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetDestinationRectangle failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_surfaceSetSourceRectangle(id, 0, 0, width, height))
        LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetSourceRectangle failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_surfaceSetVisibility(id, ILM_TRUE))
        LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetVisibility failed for surface %d ", __FUNCTION__, id));
    if (ILM_SUCCESS != ilm_layerAddSurface(mLayerId,id))
        LOG_ERROR((demo, "aautoLayerManager::%s() ilm_layerAddSurface failed for surface %d ", __FUNCTION__, id));
    else
        LOG_INFO((demo,"aautoLayerManager::%s() surface ID (%d) is added to layer ID (%d)", __FUNCTION__, id, mLayerId));

    ilm_commitChanges();
}

#ifndef IMX_DEMO
void aautoLayerManager::ilmSurfaceCB(t_ilm_uint id, struct ilmSurfaceProperties* sp, t_ilm_notification_mask mask)
{
    LOG_INFO((demo, "aautoLayerManager::%s() called", __FUNCTION__));

    // ILM_NOTIFICATION_CONFIGURED is not available on imx6
    if ((unsigned)mask & ILM_NOTIFICATION_CONFIGURED)
    {
        if (ILM_SUCCESS != ilm_surfaceSetDestinationRectangle(id, 0, 0, sp->origSourceWidth, sp->origSourceHeight))
            LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetDestinationRectangle failed for surface %d ", __FUNCTION__, id));
        if (ILM_SUCCESS != ilm_surfaceSetSourceRectangle(id, 0, 0, sp->origSourceWidth, sp->origSourceHeight))
            LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetSourceRectangle failed for surface %d ", __FUNCTION__, id));
        if (ILM_SUCCESS != ilm_surfaceSetVisibility(id, ILM_TRUE))
            LOG_ERROR((demo, "aautoLayerManager::%s() ilm_surfaceSetVisibility failed for surface %d ", __FUNCTION__, id));
        if (ILM_SUCCESS != ilm_layerAddSurface(DEFAULT_LAYER_ID, id))
            LOG_ERROR((demo, "aautoLayerManager::%s() ilm_layerAddSurface failed for surface %d ", __FUNCTION__, id));
        else
            LOG_INFO((demo,"aautoLayerManager::%s() surface ID (%d) is added to layer ID (%d)", __FUNCTION__, id, DEFAULT_LAYER_ID));

        ilm_commitChanges();
    }
}
#endif

} } // namespace adit { namespace aauto
